/* eslint-disable no-template-curly-in-string */
/**
 * @module archive
 */

let $$ = require('cdev/debug')({debug: 'cache', path: module.id, prefix: '[isg.xpert.archive]'});
$$.mdb = require('/agorum/roi/customers/cdev.ncore/js/utils/mdb')($$);
$$.pdf = require('/agorum/roi/customers/cdev.ncore/js/utils/pdf')($$);
$$.md = require('/agorum/roi/customers/cdev.ncore/js/utils/md')($$);
$$.objects = require('common/objects');

let metadb = require('common/metadb');
let templates = require('common/templates');
let mc = require('common/metadata-collection');
let time = require('common/time');

let utils = require('./idx-utils');

const RECORD_COUNTER = 'MAIN_MODULE_MANAGEMENT/customers/isg.xpert.archive/counter/records';
const DOCUMENT_COUNTER = 'MAIN_MODULE_MANAGEMENT/customers/isg.xpert.archive/counter/documents';
const CONFIG = 'MAIN_MODULE_MANAGEMENT/customers/isg.xpert.archive/config';

// load config settings from metadb
let config = $$.mdb.load(CONFIG);

// variable aliases for config settings
let PREFIX = config.prefix;
// let IDX_CHARSET = config.charset;
let BLOCK_SIZE = config.blocksize;
let BLOCK_COUNT = config.blockcount;

const RECORD_PROPERTIES = mc.get(PREFIX + 'Record', null).items.map(item => item.name);
// const DOCUMENT_PROPERTIES = mc.get(PREFIX + 'Document', null).items.map(item => item.name);

// const IMPORT = $$.objects.find(1255164); // debug/testdaten_xpert_qs
// const ERRORS = $$.objects.find(1276788); // workspace/import_errors

/**
 * Processes a document and archives it.
 *
 * @param {object} idx - The document to process.
 * @returns {object} The archived document.
 * @throws {Error} If the image name is invalid or the image file is not found.
 */
const processDocument = (idx, meta) => {
  let query = 'inpath:9999 area:mdb area:archive identifier:';
  let records = $$.objects.query(query + 'records').find()[0];
  if (!records) throw new Error('missing archive directory [records]');

  // NO DOCUMENTS FOLDER IN ARCHIVE
  // let documents = $$.objects.query(query + 'documents').find()[0];
  // if (!documents) throw new Error('missing archive directory [documents]');

  ////////////////////////////////////////////////////
  // all files found, metadata from idx file loaded //
  ////////////////////////////////////////////////////

  // search for insurance record by insNumber
  query = 'inpath:{0} {1}insnumber:{2}'.format(records.ID, PREFIX, meta[PREFIX + 'insNumber']);
  $$.debug('processDocument - query archive:', query);
  let record = $$.objects.query(query).find()[0];
  if (record) {
    meta[PREFIX + 'recordCounter'] = $$.md.load(record)[PREFIX + 'recordCounter'];
    $$.debug('processDocument - found record [{0}]:'.format(meta[PREFIX + 'recordCounter']), record);
  } else {
    meta[PREFIX + 'recordCounter'] = metadb.inc(RECORD_COUNTER, 0);
    $$.debug('processDocument - new record [{0}]:'.format(meta[PREFIX + 'recordCounter']), record);
  }
  meta[PREFIX + 'documentCounter'] = metadb.inc(DOCUMENT_COUNTER, 0);

  $$.debug(
    'processDocument - counter: document={0} record={1}'.format(
      meta[PREFIX + 'documentCounter'],
      meta[PREFIX + 'recordCounter']
    )
  );

  // NO DOCUMENTS FOLDER IN ARCHIVE
  // let docPath = utils.archivePath(meta[PREFIX + 'documentCounter'], {
  //   size: BLOCK_SIZE,
  //   count: BLOCK_COUNT,
  //   trunc: true,
  // });

  let recPath =
    utils.archivePath(meta[PREFIX + 'recordCounter'], {
      size: BLOCK_SIZE,
      count: BLOCK_COUNT,
      trunc: true,
    }) +
    templates.fill(
      '/${isg_xpert_archive_insNumber} - ${isg_xpert_archive_insName} - ${isg_xpert_archive_insBirthday:de|dd.MM.yyyy}',
      meta
    );

  // create record directory and set metadata
  let recTarget = records.createPath(recPath);

  let parent = $$.objects.find(recTarget.firstParent);
  $$.md.save(parent, { isg_xpert_archive_documentFolderType: 'records' });

  let recMeta = {};
  RECORD_PROPERTIES.forEach(prop => {
    recMeta[prop] = meta[prop];
  });
  recMeta['ag_metadata_collection'] = 'isg_xpert_archive_Record';
  $$.debug('processDocument - metadata (record)', recMeta);
  $$.md.save($$.objects.find(recTarget), recMeta);

  // create document subdirectories
  let recDocPath = templates.fill(
    '${isg_xpert_archive_processLvl2}/${isg_xpert_archive_orderNumber}/${isg_xpert_archive_processLvl3}',
    meta
  );
  let recDocTarget = recTarget.createPath(recDocPath);
  $$.md.save(recDocTarget, { isg_xpert_archive_documentFolderType: 'recordDocs' });


  // NO DOCUMENTS FOLDER IN ARCHIVE
  // let docTarget = documents.createPath(docPath);
  // $$.md.save(docTarget, { isg_xpert_archive_documentFolderType: 'documents' });

  meta['ag_metadata_collection'] = 'isg_xpert_archive_Document';
  $$.debug('processDocument - metadata (document)', meta);

  // NO DOCUMENTS FOLDER IN ARCHIVE
  // utils.moveImages(idx, meta, [recDocTarget, docTarget], true);
  utils.moveImages(idx, meta, [recDocTarget], true);
  $$.objects.unlink(idx);

  $$.debug('processDocument - archive:', recTarget.anyFolderPath.split('/').slice(-5));
  $$.log('insurance record [{0}] processed'.format(recPath));

  // return img;
};

const updateForm = (obj, collection, form) => {
  let meta = $$.md.load(obj);
  mc.get(PREFIX + collection, null).items.forEach(item => {
    let widget = form.down(item.name);
    if (widget) {
      widget.value =
        item.type === 'date' ? templates.fill('${_date:de|dd.MM.yyyy}', { _date: meta[item.name] }) : meta[item.name];
      // $$.debug('updateForm: {0} = {1}'.format(item.name, widget.value));
    }
  });
};

const exportDocuments = (id, filter) => {

  $$.debug('exportDocuments', id, filter);

  let obj = $$.objects.tryFind(id);
  if (obj) {
    id = $$.md.load(obj, new RegExp('^{0}'.format(PREFIX)))[PREFIX + 'insNumber'];
  }
  $$.debug('exportDocuments - id', id);

  let outputTarget = $$.objects
    .query('inpath:${FOLDERPATH:/agorum/roi/Files} area:mdb area:export identifier:export')
    .find()[0];
  if (!outputTarget) throw new Error('export directory not found');
  $$.debug('exportDocuments - outputTarget', outputTarget);

  let query = 'inpath:${FOLDERPATH:/agorum/roi/Files} area:mdb area:archive area:records {0}:{1}'.format(
    PREFIX + 'insnumber',
    id
  );
  let record = $$.objects.query(query).limit(1).find()[0];
  if (!record) throw new Error('invalid insurance id [{0}]'.format(id));

  let merged = $$.objects.create('file', {
    name: templates.fill('${id}_${stamp:de|YYYY-MM-dd_HHmmss}.pdf', { id: id, stamp: new Date() }),
    target: outputTarget,
  });
  $$.debug('exportDocuments - merged', merged);

  // search pdf files for insurance id with valid pdf validation (or no validation at all)
  query += ' classname:FILEOBJECT nameextension:pdf NOT isg_xpert_archive_pdfvalidation:false';
  // optional add filter from export call
  if (filter) {
    query += ' ' + filter;
  }
  let count = $$.objects.query(query).limit(0).search('id').total;
  $$.debug('exportDocuments - query', count, query);

  let numberOfDocuments = 0;
  let numberOfPages = 0;

  merged = $$.pdf.mergeDocumentsFromQuery(merged, {
    query: query,
    sort: ['isg_xpert_archive_processLvl2', 'isg_xpert_archive_orderNumber', 'isg_xpert_archive_processLvl3', 'name'],
    error: (object, e) => {
      $$.log('ERROR [{0}] for [{1}] at [{2}]'.format(e.message, object.name, record.name));
      $$.debug('exportDocuments - error', object, e, record);
    },
    success: (object, pdDocument) => {
      numberOfDocuments++;
      numberOfPages += pdDocument.getNumberOfPages();
      $$.debug('exportDocuments - success', object, pdDocument.getNumberOfPages());
    },
  });
  $$.debug('exportDocuments - merged', merged, numberOfDocuments, numberOfPages);

  let ok = $$.objects.create('file', {
    name: merged.name + '.ok',
    target: outputTarget,
  });

  let expire = time.fromDateObject(new Date()).addHours(24);
  ok.expirationDate = expire.getDateObject();
  merged.expirationDate = expire.addMinutes(30).getDateObject();

  return merged;
};

module.exports = {
  processDocument: processDocument,
  updateForm: updateForm,
  exportDocuments: exportDocuments,
};
